/**
 * \file: sdc_helper.h
 *
 * \brief : Definition of SDC helper functions mainly used by sdc_ctl and tests
 * <b>Normal applications should not require these functions </b>
 *
 * \author: Christoph Gellner (cgellner@de.adit-jv.com)
 *
 * \copyright (c) 2015 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/

/**
 * \defgroup helper Helper functions
 * \ingroup add_libSDC_API
 * Definition of SDC helper functions mainly used by sdc_ctl and tests.
 */

#ifndef __SDC_LIB_HELPER_H_
#define __SDC_LIB_HELPER_H_

#include <sdc_op_common.h>

/* Definitions types and defaults */

/**
 * \ingroup helper
 * \brief Used to specify that this function is part of the helper API of libSDC
 *
 * Usual applications using libSDC won't use these functions.
 * Anyway these functions need to be visible for applications like sdc_ctl which
 * need to iterate over the different mechanisms */
#define SDC_HELP_API __attribute__((__visibility__("default")))

/**
 * \ingroup helper
 * \brief select encrypt/decrypt, wrap/unwrap, sign/verify or digest
 *
 * Select if you want to iterate (see \ref sdc_helper_mechanism_iterator_init) or
 * set the name (see \ref sdc_helper_set_mechanism_by_name) for
 *  - encrypt/decrypt
 *  - wrap/unwrap
 *  - sign/verify
 */
typedef enum {
    SDC_ENCRYPT_DECRYPT,
    SDC_WRAP_UNWRAP,
    SDC_SIGN_VERIFY,
    SDC_DGST
} sdc_mechanism_kinds_t;

/**
 * \ingroup helper
 * \brief internal type to define the callback functions for different \ref sdc_mechanism_kinds_t
 */
typedef struct sdc_operation_info_funcs sdc_operation_info_funcs_t;

/**
 * \ingroup helper
 * \brief type to keep track of the current position of the iterator
 */
typedef struct {
    const sdc_operation_info_funcs_t *funcs; /**< variable to store the function pointers */
    void *type_specific_iterator; /**< pointer to internal type storing the position of the iterator */
} sdc_mechanism_iterator_t;


/* Functions */

/**
 * \ingroup helper
 * \brief Initialize the iterator
 *
 * \param[in]   kind select encrypt/decrypt, wrap/unwrap or sign/verify
 * \param[out]  iter iterator to be initialized
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_ALG_MODE_INVALID - the selected \ref sdc_mechanism_kinds_t is empty
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_HELP_API sdc_error_t sdc_helper_mechanism_iterator_init(sdc_mechanism_kinds_t kind, sdc_mechanism_iterator_t *iter);

/**
 * \ingroup helper
 * \brief Proceed to next

 * \param[in,out]  iter iterator
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_ALG_MODE_INVALID - no more elements
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_HELP_API sdc_error_t sdc_helper_mechanism_iterator_next(sdc_mechanism_iterator_t *iter);

/**
 * \ingroup helper
 * \brief Cleanup iterator
 *
 * \param[in] iter iterator
 *
 * \return \ref SDC_OK - no error
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_HELP_API void sdc_helper_mechanism_iterator_cleanup(sdc_mechanism_iterator_t *iter);

/**
 * \ingroup helper
 * \brief Set \p allocated_type according to the current alg/blk (alg/hash) of the iterator
 *
 * \p allocated_type has to be allocated externally.
 * The type depends on the \p kind of \ref sdc_mechanism_kinds_t
 *  - encrypt/decrypt   : \ref sdc_encrypt_decrypt_type_t
 *  - wrap/unwrap       : \ref sdc_wrap_unwrap_type_t
 *  - sign/verify       : \ref sdc_sign_verify_type_t
 *
 * \param[in] iter iterator
 * \param[out] allocated_type externally allocated type to fill
 *
 * \return \ref SDC_OK - no error
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_HELP_API sdc_error_t sdc_helper_mechanism_set_type(sdc_mechanism_iterator_t *iter, void *allocated_type);

/**
 * \ingroup helper
 * \brief Get the name of the current alg/blk (alg/hash) of the iterator
 *
 * \param[in] iter iterator
 * \param[out] name char pointer
 * \param[in] name_len max number of characters to be stored to \p name
 *
 * \return \ref SDC_OK - no error
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_HELP_API sdc_error_t sdc_helper_mechanism_get_name(sdc_mechanism_iterator_t *iter, char* name, size_t name_len);

/**
 * \ingroup helper
 * \brief Get the key format/length supported by the current alg/blk (alg/hash) of the iterator
 *
 * Note: in case returned fmt is a public key format the corresponding
 *       private key format can be used as well
 *
 * In order to convert this information to string \ref sdc_key_fmt_get_name and \ref sdc_helper_key_len_str
 *
 * \param[in] iter iterator
 * \param[out] sup_key_fmt_protect      if != NULL fill set supported key format - unprotect operation (sign, encrypt, wrap)
 * \param[out] sup_key_fmt_unprotect    if != NULL fill set supported key format - unprotect operation (verify, decrypt, unwrap)
 * \param[out] sup_key_lens             if != NULL fill bmsk with all supported key length
 * \param[out] dflt_key_len             if != NULL fill set default key length
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_KEYLEN_INVALID - operation does not use keys and is therefore independent of key-length
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_HELP_API sdc_error_t sdc_helper_mechanism_get_key_info(sdc_mechanism_iterator_t *iter,
                                                           sdc_key_fmt_t *sup_key_fmt_protect, sdc_key_fmt_t *sup_key_fmt_unprotect,
                                                           sdc_key_len_bmsk_t *sup_key_lens, sdc_key_len_t *dflt_key_len);

/**
 * \ingroup helper
 * \brief Convert the string name to the desired type (\ref sdc_encrypt_decrypt_type_t, \ref sdc_wrap_unwrap_type_t, \ref sdc_sign_verify_type_t)
 *
 * \p allocated_type has to be allocated externally.
 * The type depends on the \p kind of \ref sdc_mechanism_kinds_t
 *  - encrypt/decrypt   : \ref sdc_encrypt_decrypt_type_t
 *  - wrap/unwrap       : \ref sdc_wrap_unwrap_type_t
 *  - sign/verify       : \ref sdc_sign_verify_type_t
 *
 * \param[in]  kind select encrypt/decrypt, wrap/unwrap or sign/verify
 * \param[in]  search_name string to specify the desired type
 * \param[out] allocated_type externally allocated type to fill
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_ALG_MODE_INVALID - \p search_name could not be resolved
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_HELP_API sdc_error_t sdc_helper_set_mechanism_by_name(sdc_mechanism_kinds_t kind, char *search_name, void *allocated_type);

/**
 * \ingroup helper
 * \brief Convert \ref sdc_key_len_bmsk_t to string
 *
 * \param[in] sup_key_lens \ref sdc_key_len_bmsk_t to convert
 * \param[in] dflt_key_len default key length (marked by *) - use SDC_KEY_LEN_UNKNOWN if you don't want to mark the default
 * \param[out] keylen_info char pointer
 * \param[in] keylen_info_len max number of characters to be stored to \p keylen_info
 *
 * \return \ref SDC_OK - no error
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_HELP_API sdc_error_t sdc_helper_key_len_str(sdc_key_len_bmsk_t sup_key_lens, sdc_key_len_t dflt_key_len, char *keylen_info, size_t keylen_info_len);

/**
 * \ingroup helper
 * \brief Convert \ref sdc_encrypt_decrypt_type_t to string
 *
 * \param[in] type type
 * \param[out] typestr char pointer
 * \param[in] typestr_len max number of characters to be stored to \p typestr
 *
 * \return \ref SDC_OK - no error
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_HELP_API sdc_error_t sdc_helper_encrypt_decrypt_type_name(const sdc_encrypt_decrypt_type_t *type, char* typestr, size_t typestr_len);

/**
 * \ingroup helper
 * \brief Convert \ref sdc_wrap_unwrap_type_t to string
 *
 * \param[in] type type
 * \param[out] typestr char pointer
 * \param[in] typestr_len max number of characters to be stored to \p typestr
 *
 * \return \ref SDC_OK - no error
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_HELP_API sdc_error_t sdc_helper_wrap_unwrap_type_name(const sdc_wrap_unwrap_type_t *type, char* typestr, size_t typestr_len);

/**
 * \ingroup helper
 * \brief Convert \ref sdc_sign_verify_type_t to string
 *
 * \param[in] type type
 * \param[out] typestr char pointer
 * \param[in] typestr_len max number of characters to be stored to \p typestr
 *
 * \return \ref SDC_OK - no error
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_HELP_API sdc_error_t sdc_helper_sign_verify_type_name(const sdc_sign_verify_type_t *type, char* typestr, size_t typestr_len);

/**
 * \ingroup helper
 * \brief Convert \ref sdc_dgst_type_t to string
 *
 * \param[in] type type
 * \param[out] typestr char pointer
 * \param[in] typestr_len max number of characters to be stored to \p typestr
 *
 * \return \ref SDC_OK - no error
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_HELP_API sdc_error_t sdc_helper_dgst_type_name(const sdc_dgst_type_t *type, char* typestr, size_t typestr_len);

#endif
